function [output] = eph_analysis(vol)
%A script for ephrin data anlaysis
%Synatax:   [data] = eph_analysis(vol);
%Input:     vol = normalizing volume
%Output:    output = the data set. Sturucture with fields:...

if nargin==0    %no input
    vol = [];
end
sdist = 10;  %Default Initialized.
x_scale = 1;    %base scale of a unit in the x axis
y_scale = 1;    %base scale of a unit in the y axis
z_scale = 1;    %base scale of a unit in the z axis
noself = 0;

[ori_files,ori_dir,filterindex] = uigetfile2({'*.csv','Text files (*.csv)';'*.*','All Files';},...
    'Open ori files','Multiselect','on');
[termi_files,termi_dir,filterindex] = uigetfile2({'*.csv','Text files (*.csv)';'*.*','All Files';},...
    'Open termi files','Multiselect','on');
if ~iscell(ori_files)       %make sure filenames is a cell array
    ori_files = ori_files{ori_files};
end
if ~iscell(termi_files)       %make sure filenames is a cell array
    termi_files = termi_files{termi_files};
end

%Now sort through the directory and find the unique files
[ori_names] = dir_sort(ori_files);
[termi_names] = dir_sort(termi_files);
%multi
parpool
%Process the files for size and colocalization - for ephrine termi centric
h = waitbar(0,'Analyzing Channel: ','position',[10 50 275 50]);
for i = 1:size(termi_files,2)
    waitbar((i/size(termi_files,2)),h,['Analyzing Channel: ',termi_names{i}]);   %update progress
    %open the termi files
    termi_tmp = single(dlmread([termi_dir,filesep,termi_files{i}],',',1,0));    %open the file
    %now go through the ori files
    count_data = []; %initialize
    density_data = [];   %init
    ncount_data = [];
    h2 = waitbar(0,'Comparing to Channel: ');
    for j = 1:size(ori_files,2)
        waitbar((j/size(ori_files,2)),h2,['Comparing to Channel: ',ori_names{j}]);   %update progress
        %open ori files
        ori_tmp = single(dlmread([ori_dir,filesep,ori_files{j}],',',1,0));    %open the file
        %get the unique # of ori - consider NaN removal another day - this
        %is decent enough
        size_data(j,1) = size(unique(ori_tmp,'rows'),1);
        %calculate distance metric
        [ori,termi] = distmetric_loco(ori_tmp,termi_tmp,sdist,x_scale,y_scale,z_scale,noself);
        %calculate volume normalization
        count_tmp = hist(cell2mat(ori.alldist),50);     %bin it
        density_tmp = count_tmp./vol;   %normalize it
        ncount_tmp = hist(ori.ndist,50);     %Get the nearest too
        %store together for output
        count_data = vertcat(count_data,count_tmp);
        density_data = vertcat(density_data,density_tmp);
        ncount_data = vertcat(ncount_data,ncount_tmp);
    end
    close(h2)
    %save the files
    mkdir(ori_dir,['analysis_',termi_names{i}]);
    size_out = dataset(size_data,'obsname',ori_names);
    sav2csv(size_out,'channel_size_data.csv',[ori_dir,'analysis_',termi_names{i}]);
    count_out = dataset(count_data,'obsname',ori_names);
    sav2csv(count_out,'allcount_data.csv',[ori_dir,'analysis_',termi_names{i}]);
    density_out = dataset(density_data,'obsname',ori_names);
    sav2csv(density_out,'density_data.csv',[ori_dir,'analysis_',termi_names{i}]);
    ncount_out = dataset(ncount_data,'obsname',ori_names);
    sav2csv(ncount_out,'ncount_data.csv',[ori_dir,'analysis_',termi_names{i}]);
    output(i).size = size_data;
    output(i).count = count_data;
    output(i).density = density_data;
end

close(h)
delete(gcp('nocreate'))

%-----------------------------------------------------------------------------------------------------------------------------
function [f_tmp] = dir_sort(filenames)
%give the directory and it will sort out the files in it and put a cell
%array of filenames

for i = 1:size(filenames,2)      %step through each filename and pull the wanted word
    if strcmp(filenames{i}(1,end-3:end),'.csv')     %we only want csv files
        filename_tmp = filenames{i};
        strmask = isstrprop(filename_tmp,'punct');  %find the punctuations, we only want the last two
        strmask2 = filename_tmp=='&';       %we are going to exempt the & character
        strmask3 = isstrprop(filename_tmp,'wspace');    %included the spaces as well
        strmask = strmask-strmask2+strmask3;     %removed and append
        [x,y] = find(strmask==1);  %get the positions
        a = 1;  %initiate
        b = 0;
        tmp = filename_tmp(1,y(end-a)+1:y(end-b)-1);    %create an array of the words
        while ~isempty(str2num(tmp))   %not empty = numbers or vertices file, push forward one and go
            a = a+1;
            b = b+1;
            if size(y,2)-a==0
                break
            end
            tmp = filename_tmp(1,y(end-a)+1:y(end-b)-1);    %create an array of the words
        end
        %now do the same for the front
        a = 0;
        tmp = filename_tmp(1,1:y(1+a)-1);
        while ~isempty(str2num(tmp))&&size(y,2)~=a+1   %not empty = numbers or vertices file, push forward one and go
            a = a+1;
            tmp = filename_tmp(1,a:y(1+a)-1);    %create an array of the words
        end
        if a==0
            f_tmp{i} = filename_tmp(1,y(1):y(end-b)-1);
        else
            f_tmp{i} = filename_tmp(1,y(a)+1:y(end-b)-1);
        end
    end
end
%--------------------------------------------------------------------------
function [orig,term] = distmetric_loco(ori,termi,sdist,x_scale,y_scale,z_scale,noself)
%This function calculates the distance metric.  It will use the ori
%centroids to search for all of the centroids in termi that falls within
%the search volume.
%Synatax:   [ori,termi] = distmetric('sdist',20); 
%Input:     This function does not really require any inputs.  However if
%           you do not want default values then.
%           'sdist' = the search distance.  Default = 10 pixels
%           'x' = the scale, as a ratio or xyz, of 1 unit in the x axis.
%                 Default = 1
%           'y' = the scale, as a ratio or xyz, of 1 unit in the y axis.
%                 Default = 1
%           'z' = the scale, as a ratio or xyz, of 1 unit in the z axis.
%                 Default = 1
%           'noself' = special case if you want to compare a list to
%                   itself, but don't want all points to be 0.  So, the
%                   nearest point not self.  Default = 0(off)
%Output:    The structure of the data set is this:
%           origin:
%             orig.verts = An x by 3 array of vertices that was
%               used for generating the distances.
%             orig(1).ndist = an x by 1 array of the nearst distance termi
%               vertex. (2) for random verts
%             orig(1).alldist = an x by 1 array of all of the distances of
%               termi with search distance of vertex.  (2) for random verts
%             orig.filnames = the files opened
%             orig.pathnames = the place where those files were...grammer?
%           termini:
%             term(1).nverts = an x by 3 array of the vertices in termi
%               nearest to the ori vertex. (2) for random verts
%             term(1).allverts = an x by 3 array of the vertices in termi of all
%               vertices within search distance of ori vertex. (2) for random verts 
%             term.randverts = an x by 3 array of the random vertices
%               generated with similar density.
%             term.verts = an x by 3 array of termi vertices.

multi = 1;
%now lets calculate the distance metric
[ndist,nverts,alldist,allverts] = distance_cal(ori,termi,sdist,x_scale,y_scale,z_scale,noself,multi,'ori');
%store the data
orig.ndist = ndist;
orig.alldist = alldist;
term.nverts = nverts;
term.allverts = allverts;

%--------------------------------------------------------------------------
%subfunction to calculate the distance metric.
function [ndist,nverts,alldist,allverts] = distance_cal(ori,termi,sdist,x_scale,y_scale,z_scale,noself,multi,text)

%now lets calculate the distance metric
verti_tmp = ori(:,1:3).*repmat([x_scale y_scale z_scale],size(ori,1),1);   %pull data and scale the vertices to the proper unit
%     if size(ori,2)>3      %yes extra data
%         pdata_tmp = single(ori(:,4:end));      %extra data for each vertices
%     else        %no extra data
%         pdata_tmp = zeros(size(ori,1),1);      %a zero substitute
%     end
verti_co_tmp = termi(:,1:3).*repmat([x_scale y_scale z_scale],size(termi,1),1);   %scale the vertices to the proper unit
if size(termi,2)>3      %yes extra data
    pdata_co_tmp = termi(:,4:end);     %matching brightness data for comparision channel, now include other properties
else        %no extra data
    pdata_co_tmp = zeros(size(termi,1),1);      %a zero substitute
end
%preload cell arrays
ndist = zeros(size(verti_tmp,1),1);       %shortest distances
%verti_match = cell(size(verti_tmp,1),1,size(filenames,2)); %the matching original verticies for the closeset points.
%v_match_lum = cell(size(verti_tmp,1),1,size(filenames,2)); %the matching luminance values
%dist_verti_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %the matching distance matrix to the closest point list. A cell array as well.
%closest_tmp = cell(size(verti_tmp,1),1,size(filenames,2));  %the closest pivot.  A cell array.
%closest_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));  %the matching lumninace data.  A cell array
nverts = zeros(size(termi));       %the termi vertices
%termi_prop_tmp = zeros(size(verti_tmp,1),size(bright_tmp,2),size(filenames,2)); %the termi properties
allverts = cell(size(verti_tmp,1),1);    %The short list.  A cell Array.
%short_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));    %The short list luminance.  A cell array.
alldist = cell(size(verti_tmp,1),1);     %The short list distances.  A cell Array.
%sl_verti_match_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %The matching verticies for the short list points.
%sl_v_match_lum_tmp = cell(size(verti_tmp,1),1,size(filenames,2));   %The matching luminance data.
iter = size(verti_tmp,1);   %number of iterations
if ~multi
    %start progress bar
    h = waitbar(0,[text,': Vertex Distance Analyzed: 0']);    %initialize progress bar.
    for k = 1:iter     %iterate through each vertex of the set
        %sdist is modifed to increase the search area as an
        %efficent and easy way to find points in a sphere...how...wait
        [short_list short_list_lum] = find_neighbors(verti_tmp(k,:),verti_co_tmp,sdist*2,pdata_co_tmp);    %get a short list of local pivots
        curr_verti = repmat(verti_tmp(k,:),size(short_list,1),1);         %replicate the current vertex to the size of the short list.
        if noself       %special case not self
            [y,x] = find(short_list(:,1)==curr_verti(:,1)&short_list(:,2)==curr_verti(:,2)&short_list(:,3)==curr_verti(:,3));  %find self vertex
            short_list(y,:) = [];       %gone
            short_list_lum(y,:) = [];   %gone
            curr_verti(y,:) = [];       %gone
        end
        verti_dist = single(dddist(curr_verti,short_list));     %calculate the distance of the current vertex from all points in the short list.
        %now we are going to make this a search in a sphere
        idx = verti_dist>sdist;     %any point actually outside of the specified radius
        verti_dist(idx) = [];       %gone
        short_list(idx,:) = [];     %gone
        short_list_lum(idx,:) = []; %gone
        if ~isempty(short_list)           %make sure there is something in the neighborhood.
            %curr_verti(idx,:) = [];     %gone
            [x,y] = find(verti_dist==absmin(verti_dist));   %find the position of the closest point
            %closest_tmp{k,1,j} = short_list(x,:);           %create a cell array of all the closest points, just in case of multiples.
            %closest_lum_tmp{k,1,j} = short_list_lum(x,:);   %create a matching cell array of luminance
            if size(termi,2)>3
                nverts(k,:) = [short_list(x(1,1),:) short_list_lum(x(1,1),:)];        %for now all I care about is the first closest point, switch to closest_tmp for more complete view
                allverts{k,1} = [short_list short_list_lum];    %output the short list as well.
            else
                nverts(k,:) = [short_list(x(1,1),:)];        %for now all I care about is the first closest point, switch to closest_tmp for more complete view
                allverts{k,1} = [short_list];    %output the short list as well.
            end
            %verti_match{k,1,j} = repmat(verti_tmp(k,:),size(short_list(x,:),1),1);    %create a matching original verticies for the closets pts.
            %v_match_lum{k,1,j} = repmat(bright_tmp(k,:),size(short_list(x,:),1),1);
            %dist_verti_tmp{k,1,j} = verti_dist(x,:);  %create a matching set of distances with the vertices.
            ndist(k,1) = verti_dist(x(1,1),:);              %create a columnar vector of shortest distances
            %vert_prop(k,:) = bright_tmp(k,:);         %store the vertices associated properties
            alldist{k,1} = verti_dist;  %output the distances for all points in the short list.
            %sl_verti_match_tmp{k,1,j} = repmat(verti_tmp(k,:),size(short_list,1),1);  %create a matching vertices for the short list
            %sl_v_match_lum_tmp{k,1,j} = repmat(bright_tmp(k,:),size(short_list,1),1);  %create a matching luminance for the short list
        else                        %there is no one around me (sad)
            %closest_tmp{k,1,j} = single([NaN NaN NaN]);           %not a number for easy extraction.
            %closest_lum_tmp{k,1,j} = single(NaN(1,size(bright_tmp,2)));
            if size(termi,2)>3
                nverts(k,:) = [single([NaN NaN NaN]) single(NaN(1,size(pdata_co_tmp,2)))];
                allverts{k,1} = [single([NaN NaN NaN]) single(NaN(1,size(pdata_co_tmp,2)))];    %output the short list as well.
            else
                nverts(k,:) = [single([NaN NaN NaN])];
                allverts{k,1} = [single([NaN NaN NaN])];    %output the short list as well.
            end
            %verti_match{k,1,j} = verti_tmp(k,:);
            %v_match_lum{k,1,j} = bright_tmp(k,:);
            %dist_verti_tmp{k,1,j} = single(NaN);
            ndist(k,1) = single(NaN);
            %vert_prop(k,:,j) = single(NaN(1,size(bright_tmp,2)));
            %short_lum_tmp{k,1,j} = single(NaN);
            alldist{k,1} = single(NaN);   %output the distance for short list.
            %sl_verti_match_tmp{k,1,j} = verti_tmp(k,:);  %all by yourself
            %sl_v_match_lum_tmp{k,1,j} = bright_tmp(k,:);
        end
        waitbar(k/iter,h,[text,': Vertex Distance Analyzed: ',num2str(k)]);   %update progress
    end
    close(h);   %close progress bar
end
if multi
    parfor k = 1:iter     %iterate through each vertex of the set
        %sdist is modifed to increase the search area as an
        %efficent and easy way to find points in a sphere...how...wait
        [short_list short_list_lum] = find_neighbors(verti_tmp(k,:),verti_co_tmp,sdist*2,pdata_co_tmp);    %get a short list of local pivots
        curr_verti = repmat(verti_tmp(k,:),size(short_list,1),1);         %replicate the current vertex to the size of the short list.
        if noself       %special case not self
            [y,x] = find(short_list(:,1)==curr_verti(:,1)&short_list(:,2)==curr_verti(:,2)&short_list(:,3)==curr_verti(:,3));  %find self vertex
            short_list(y,:) = [];       %gone
            short_list_lum(y,:) = [];   %gone
            curr_verti(y,:) = [];       %gone
        end
        verti_dist = single(dddist(curr_verti,short_list));     %calculate the distance of the current vertex from all points in the short list.
        %now we are going to make this a search in a sphere
        idx = verti_dist>sdist;     %any point actually outside of the specified radius
        verti_dist(idx) = [];       %gone
        short_list(idx,:) = [];     %gone
        short_list_lum(idx,:) = []; %gone
        if ~isempty(short_list)           %make sure there is something in the neighborhood.
            %curr_verti(idx,:) = [];     %gone
            [x,y] = find(verti_dist==absmin(verti_dist));   %find the position of the closest point
            %closest_tmp{k,1,j} = short_list(x,:);           %create a cell array of all the closest points, just in case of multiples.
            %closest_lum_tmp{k,1,j} = short_list_lum(x,:);   %create a matching cell array of luminance
            if size(termi,2)>3
                nverts(k,:) = [short_list(x(1,1),:) short_list_lum(x(1,1),:)];        %for now all I care about is the first closest point, switch to closest_tmp for more complete view
                allverts{k,1} = [short_list short_list_lum];    %output the short list as well.
            else
                nverts(k,:) = [short_list(x(1,1),:)];        %for now all I care about is the first closest point, switch to closest_tmp for more complete view
                allverts{k,1} = [short_list];    %output the short list as well.
            end
            %verti_match{k,1,j} = repmat(verti_tmp(k,:),size(short_list(x,:),1),1);    %create a matching original verticies for the closets pts.
            %v_match_lum{k,1,j} = repmat(bright_tmp(k,:),size(short_list(x,:),1),1);
            %dist_verti_tmp{k,1,j} = verti_dist(x,:);  %create a matching set of distances with the vertices.
            ndist(k,1) = verti_dist(x(1,1),:);              %create a columnar vector of shortest distances
            %vert_prop(k,:) = bright_tmp(k,:);         %store the vertices associated properties
            alldist{k,1} = verti_dist;  %output the distances for all points in the short list.
            %sl_verti_match_tmp{k,1,j} = repmat(verti_tmp(k,:),size(short_list,1),1);  %create a matching vertices for the short list
            %sl_v_match_lum_tmp{k,1,j} = repmat(bright_tmp(k,:),size(short_list,1),1);  %create a matching luminance for the short list
        else                        %there is no one around me (sad)
            %closest_tmp{k,1,j} = single([NaN NaN NaN]);           %not a number for easy extraction.
            %closest_lum_tmp{k,1,j} = single(NaN(1,size(bright_tmp,2)));
            if size(termi,2)>3
                nverts(k,:) = [single([NaN NaN NaN]) single(NaN(1,size(pdata_co_tmp,2)))];
                allverts{k,1} = [single([NaN NaN NaN]) single(NaN(1,size(pdata_co_tmp,2)))];    %output the short list as well.
            else
                nverts(k,:) = [single([NaN NaN NaN])];
                allverts{k,1} = [single([NaN NaN NaN])];    %output the short list as well.
            end
            %verti_match{k,1,j} = verti_tmp(k,:);
            %v_match_lum{k,1,j} = bright_tmp(k,:);
            %dist_verti_tmp{k,1,j} = single(NaN);
            ndist(k,1) = single(NaN);
            %vert_prop(k,:,j) = single(NaN(1,size(bright_tmp,2)));
            %short_lum_tmp{k,1,j} = single(NaN);
            alldist{k,1} = single(NaN);   %output the distance for short list.
            %sl_verti_match_tmp{k,1,j} = verti_tmp(k,:);  %all by yourself
            %sl_v_match_lum_tmp{k,1,j} = bright_tmp(k,:);
        end
    end
end